/********************************

	Code for Abramitzky, Boustan, Eriksson (2019)
	
		"Do Immigrants Assimilate More Slowly Today than in the Past?"
		American Economic Review: Insights

		This do-file prepares results using data from California (1989-2015)
		
	Note: Because of computational constraints, we load and prep the dataset separately each time
	
	
*********************************/

set more off
clear
clear mata
clear matrix
cap log close
set matsize 1000
set maxvar 2500


******************************************************************************

* set directories
global main ""
global data "$main/1data"
global work "$main/3work"

sysdir set PLUS "E:/Stata/plus"
sysdir set PERSONAL "E:/Stata/personal"

log using "$work/AERI_Results_$S_DATE.txt", text replace

******************************************************************************

***********************
*** Define Programs ***
***********************

capture program drop store_regular3
program define store_regular3, rclass

	args var

	local b = string(_b[`var'], "%10.3fc") 
	local se = "(" + string(_se[`var'], "%10.3fc") + ")"
	*local p = 2*ttail(e(df_r),abs(_b[`var']/_se[`var']))
	local p = 2*(1-normal(abs(_b[`var']/_se[`var'])))
	local star=""
		if `p' >.05 & `p' <=.10 { 	
			local star = "$^{*}$"
		}
		if `p' >.01 & `p' <=.05 { 	
			local star = "$^{**}$"
		}
		if `p' <=.01 {	
			local star = "$^{***}$" 
		}
	
	return local beta = "`b'"
	return local se = "`se'"
	return local star = "`star'"
	return local p = "`p'"

end	
	
		
		

capture program drop setup_main
program define setup_main, rclass

	* drop vars we don't need because of computational constraints
	cap drop freqUS freqFB ttf firstname // don't need these anymore
	cap drop mmaidenname mfirstname mbirthdate mbirthyear meduc
	cap drop maidenindex
	
	drop if mi(FBindex) // drop if missing dependent variable
	
	* option here to drop if missing total ever born, can't determine parity
	drop if totaleverborn==99 | totaleverborn==0 // one obs with 0
	
	* option to topcode remaining - makes little difference
	sum totaleverborn, d
	replace totaleverborn = `r(p99)' if totaleverborn>`r(p99)' & !mi(totaleverborn)
	
	replace FBindex = FBindex*100
	
	* birthyear in 5-year bands
	forvalues i = 2/7{
		gen band`i'=(birthyear>=1980+(`i'-1)*5 & birthyear<1980+`i'*5)
	}
	
	replace band7=1 if birthyear==2015
	
	* create foreign indicator
	gen foreignborn = (mnativity==0)
	
	* mother's age indicators
	forvalues i = 21/40{
		gen ind_mage_`i' = (mage==`i')	
		qui compress
	}
	
	* dropping nonwhite
	drop if mrace==20 & foreignborn==0
	
	qui compress
	
end	




capture program drop setup_3yr
program define setup_3yr, rclass

	* drop vars we don't need because of computational constraints
	cap drop freqUS freqFB ttf firstname // don't need these anymore
	cap drop mmaidenname mfirstname mbirthdate mbirthyear medu maidenquart
	cap drop maidenindex
	
	drop if mi(FBindex) // drop if missing dependent variable
	
	* option here to drop if missing total ever born, can't determine parity
	drop if totaleverborn==99 | totaleverborn==0 // one obs with 0
	
	* option to topcode remaining - makes little difference
	sum totaleverborn, d
	replace totaleverborn = `r(p99)' if totaleverborn>`r(p99)' & !mi(totaleverborn)
	
	replace FBindex = FBindex*100
	
	* birthyear in 5-year bands
	forvalues i = 2/12{
		gen band`i'=(birthyear>=1980+(`i'-1)*3 & birthyear<1980+`i'*3)
	}
	
	cap drop band2-band4
	
	* create foreign indicator
	gen foreignborn = (mnativity==0)
	
	* mother's age indicators - 20 is reference category
	forvalues i = 21/40{
		gen ind_mage_`i' = (mage==`i')	
		qui compress
	}
	
	* dropping nonwhite
	drop if mrace==20 & foreignborn==0
	
	qui compress
end	




************************
*** Set Figure Style ***
************************

set scheme plotplainblind

graph set eps fontface "Times New Roman"
graph set eps fontfacesans "Times New Roman"
graph set eps fontfaceserif "Times New Roman"
graph set eps fontfacemono "Times New Roman"

local aspectratio 0.5




******************************************************************
*** Figure 1: Panel B - Immigrants Selected Less Foreign Names ***
******************************************************************


	use "$work/compiled_indexalt.dta", replace
	setup_main

	* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex ind_mage_* band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figure1_fb
		
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0, abs(mgroup) vce(robust) coeflegend
		estimates store figure1_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figure1_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = _b[_cons] in 1
		replace se_native = _se[_cons] in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figure1_fb
		replace coef_foreign = _b[_cons] in 1
		replace se_foreign = _se[_cons] in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) m(o) lp(solid)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
		graph export "$work/Figures/fig1_b.eps", replace	



	
	
	

*******************************************************************************
*** Figure 2, Panel B: Results by Sending Country (Pooled Model Interacted) ***
*******************************************************************************

* labels = cons + fb indicator
* bar = 20 * \beta

use "$work/compiled_indexalt.dta", replace
	setup_main

	foreach country in MX CH PI VN RE {
		gen country_`country' = (mbirthplace=="`country'")
		gen mage_`country' = mage*country_`country'
		cap drop country_`country'
	
	}
	
	foreach country in MX CH PI VN RE{
		* first, get the mean level
		reg FBindex mage i.totaleverborn band4-band7 i.foreignborn ///
			if (foreignborn==1 & mbirthplace=="`country'") | (foreignborn==0), ///
			vce(robust)
			
			estimates store est_`country'_mean
	}
	
		* then get bar value from the pooled regression on these countries
		areg FBindex mage_RE mage_MX mage_CH mage_PI mage_VN i.totaleverborn band4-band7 ///
			if (foreignborn==1 & inlist(mbirthplace, "RE", "MX", "CH", "PI", "VN")) ///
			, abs(mgroup) ///
			vce(robust)
			
			estimates store est_bar
			
			foreach country in MX CH PI VN RE{
				test mage_`country'==0
				local p_`country' = `r(p)'
			}
			
	
	clear
	set obs 5
	gen country = ""
	gen bpl = ""
	gen mean = .
	gen bar = .
	gen p = .
	
	local i = 0
	foreach country in MX CH PI VN RE{
		local ++ i
		replace bpl = "`country'" in `i'
	}
	
	replace country = "Mexico" if bpl=="MX"
	replace country = "China" if bpl=="CH"
	replace country = "Philippines" if bpl=="PI"
	replace country = "Vietnam" if bpl=="VN"
	replace country = "Rest of World" if bpl=="RE"
	
	foreach country in MX CH PI VN RE{
		
		* estimates are from the full set of interactions
		estimates restore est_bar
		replace bar = _b[mage_`country']*20 if bpl=="`country'"
		replace p = `p_`country'' if bpl=="`country'"
		
		estimates restore est_`country'_mean
		replace mean = _b[_cons] + _b[1.foreignborn] if bpl=="`country'"
			format mean %3.0fc
		
	}	
	
	tostring mean, replace force
	replace mean = substr(mean,1,2)
	sort bar
	gen N = _n
	gen star = ""
	replace star = "*" if p<0.05
	gen label = country + "(F = " + mean + ")" + star
	
	labmask N, values(label)
	graph bar (asis) bar, over(N, label( angle(45))) ///
		ytitle( "Estimated effect of 20 additional years of" "mother's age at birth on Foreignness Index†") ///
		ylab(-15(5)0)
		
	graph export "$work/Figures/fig2_b.eps", replace

		
	
	


	
		
********************************************************************************************
********** Table 1, Panel B: Immigrant Name Choice by Sending Country x Education **********		
********************************************************************************************

	use "$work/compiled_indexalt.dta", replace
	setup_main

	* Table preliminaries
	cap erase "$work/Tables/table2.tex"
	file open table2 using "$work/Tables/table2.tex", write replace
	
		file write table2 ///
			"\begin{tabular}{l cccc}" _n ///
			"\toprule" _n ///
			"\multicolumn{5}{c}{Depdendent Variable = Foreignness Index} \\" _n ///
			"\cline{1-5}" _n ///
			"& \multicolumn{1}{c}{Less than} & \multicolumn{1}{c}{} & \multicolumn{1}{c}{} & \multicolumn{1}{c}{College}  \\" _n ///
			"& \multicolumn{1}{c}{HS} & \multicolumn{1}{c}{HS grad} & \multicolumn{1}{c}{Some College} & \multicolumn{1}{c}{plus} \\" _n ///
			"\cline{1-5}" _n ///
			"& \multicolumn{1}{c}{(1)} & \multicolumn{1}{c}{(2)} & \multicolumn{1}{c}{(3)} & \multicolumn{1}{c}{(4)}\\" _n ///
			"\cline{1-5}" _n
			
		* Panel B: Results by Sending Country x Education
		foreach country in VN MX CH {
					
			* Less than high school
			areg FBindex mage i.totaleverborn band4-band7 ///
				if ((foreignborn==1 & mbirthplace=="`country'") & (max_meduc>=0 & max_meduc<12)), ///
				abs(mgroup) vce(robust)
				
				store_regular3 mage
				local b_`country'_1 = "`r(beta)'`r(star)'"
				local se_`country'_1 = "`r(se)'" 
				local obs_`country'_1 = "\multicolumn{1}{c}{" + string(`e(N)', "%10.0fc") + "}"

			* High school grad
			areg FBindex mage i.totaleverborn band4-band7 ///
				if ((foreignborn==1 & mbirthplace=="`country'") & (max_meduc==12)), ///
				abs(mgroup) vce(robust)
				
				
				store_regular3 mage
				local b_`country'_2 = "`r(beta)'`r(star)'"
				local se_`country'_2 = "`r(se)'" 
				local obs_`country'_2 = "\multicolumn{1}{c}{" + string(`e(N)', "%10.0fc") + "}"
				
			* Some college
			areg FBindex mage i.totaleverborn band4-band7 ///
				if ((foreignborn==1 & mbirthplace=="`country'") & (max_meduc>=13 & max_meduc<=15)), ///
				abs(mgroup) vce(robust)
				
				store_regular3 mage
				local b_`country'_3 = "`r(beta)'`r(star)'"
				local se_`country'_3 = "`r(se)'"
				local obs_`country'_3 = "\multicolumn{1}{c}{" + string(`e(N)', "%10.0fc") + "}"
				
			* Bachelor's or More
			areg FBindex mage i.totaleverborn band4-band7 ///
				if ((foreignborn==1 & mbirthplace=="`country'") & (max_meduc>=16 & !mi(max_meduc))), ///
				abs(mgroup) vce(robust)
				
				store_regular3 mage
				local b_`country'_4 = "`r(beta)'`r(star)'"
				local se_`country'_4 = "`r(se)'" 
				local obs_`country'_4 = "\multicolumn{1}{c}{" + string(`e(N)', "%10.0fc") + "}"
				
		}	
	
		* Write results
		file write table2 "Vietnam & `b_VN_1' & `b_VN_2' & `b_VN_3' & `b_VN_4'  \\" _n ///
			"& `se_VN_1' & `se_VN_2' & `se_VN_3' & `se_VN_4'  \\" _n 
		
		file write table2 "\addlinespace" _n
		file write table2 "\emph{N} & `obs_VN_1' & `obs_VN_2' & `obs_VN_3' & `obs_VN_4'  \\" _n 
		file write table2 "\addlinespace" _n
	
		file write table2 "Mexico & `b_MX_1' & `b_MX_2' & `b_MX_3' & `b_MX_4'  \\" _n ///
			"& `se_MX_1' & `se_MX_2' & `se_MX_3' & `se_MX_4'  \\" _n 
		
		file write table2 "\addlinespace" _n
		file write table2 "\emph{N} & `obs_MX_1' & `obs_MX_2' & `obs_MX_3' & `obs_MX_4'  \\" _n 
		file write table2 "\addlinespace" _n		
	
		file write table2 "China & `b_CH_1' & `b_CH_2' & `b_CH_3' & `b_CH_4'  \\" _n ///
			"& `se_CH_1' & `se_CH_2' & `se_MX_3' & `se_CH_4'  \\" _n 
		
		file write table2 "\addlinespace" _n
		file write table2 "\emph{N} & `obs_CH_1' & `obs_CH_2' & `obs_CH_3' & `obs_CH_4'  \\" _n 
	
		file write table2 ///
			"\bottomrule" _n ///
			"\end{tabular}" _n
			
		file close table2
		macro drop table2
			
	
	
	

/***********************************************

		APPENDIX FIGURES AND TABLES


***********************************************/


*********************************************************************
********** Appendix Figure 1: Estimated Match Rate by Year **********
*****															*****
********** Appendix Table 1: Characteristics of (Un)Matched *********
*********************************************************************

	

	use "$main/3work/CAbirths_children_coded_indexalt.dta", replace
	
	destring mbirthyear, replace
	compress
	
	* drop black
	drop if mrace==20 & mnativity==1
	
	* 1) drop these vars for grouping
	//cap destring birthdate, replace
	drop if mmaidenname ==""
	drop if mbirthplace=="NA" | mbirthplace==""
	drop if mfirstname==""
	drop if mbirthyear==. | mi(mbirthyear)
	drop if mbirthdate=="." | mbirthdate==""
	
	drop if birthyear==2011
	
	cap drop mgroup
	egen mgroup=group(mfirstname mmaidenname mbirthdate mbirthplace mbirthyear)	
	
	* create a "matched" sample to merge back
	drop if totaleverborn==99
	bys mgroup: egen max_ever_born = max(totaleverborn)
	keep mgroup mfirstname mmaidenname mbirthdate mbirthplace mbirthyear max_ever_born
	bys mgroup: gen num_obs = _N
	duplicates drop
	
	save "$main/3work/matched_sample_for_merge_indexalt.dta", replace
	
	
	**************************
	
	use "$main/3work/CAbirths_children_coded_indexalt.dta", replace
	
	destring mbirthyear, replace
	compress
	
	* drop black
	drop if mrace==20 & mnativity==1
	
	merge m:1 mfirstname mmaidenname mbirthdate mbirthplace mbirthyear using "$main/3work/matched_sample_for_merge_indexalt.dta", assert(1 3)
	
		* NOW, start with the universe of potential matches
		* anyone who has totaleverborn>=2
		drop if totaleverborn==99
		keep if totaleverborn>=2 // this is the universe of potential matches
		
		drop if birthyear==2011

		* num obs is the count of times we observe you in the matched_sample
		gen bad_data = 0
		replace bad_data = 1 if mmaidenname =="" | (mbirthplace=="NA" | mbirthplace=="") | mfirstname=="" | (mbirthyear==. | mi(mbirthyear)) | (mbirthdate=="." | mbirthdate=="")
		
		* only observe you once even though you report having two children (these are the people we can't link)
		gen only_one_obs = (num_obs==1)
		
		* cross tab of bad data - how much is bad data contributing
		tab birthyear bad_data
		
		* cross tab of unlinked by birthyear (conditional on not having bad data)
		tab birthyear only_one_obs if bad_data==0
		
		drop if birthyear<1989 | birthyear==2011
		
		* characteristics of any unlinked (note if there are other name errors, we will never be able to know)
		gen unlinked = 0
		replace unlinked = 1 if bad_data==1 | only_one_obs==1
	
			* CA mother born
			gen ca_born = mbirthplace=="CA"
							
			* collapse and create graph
			preserve
				gen denom = 1
				collapse (sum) unlinked denom, by(birthyear)
				
				gen match_rate = 1-(unlinked/denom)

					* figure for export
					twoway (connected match_rate birthyear, mc(black) lc(black) m(o)), ///
						ytitle("Match Rate") ///
						xtitle("Birthyear") ///
						xsc(range(1988 2016)) ///
						xlab(1988(4)2016, nogrid) ///
						ylab(0.3(.10)0.9) ///
						legend(off)
						
					graph export "$work/Figures/figA1.eps", replace
			restore
			
			
	* Table comparing match characteristics
	* generate characteristics across mothers at time parity==2
	drop if bad_data==1
	gen meduc2_temp = meduc if totaleverborn==2
	gen mage2_temp = mage if totaleverborn==2
	bys mfirstname mmaidenname mbirthdate mbirthplace mbirthyear: egen meduc2 = min(meduc2_temp) //min takes care of missing values appropriately
	bys mfirstname mmaidenname mbirthdate mbirthplace mbirthyear: egen mage2 = min(mage2_temp) //min takes care of missing values appropriately
	
	* education characteristics
	gen less_than_hs = (meduc2<12) if meduc2!=99 & meduc2!=.
	gen hs = meduc2==12 if meduc2!=99 & meduc2!=.
	gen some_coll = (meduc2>=13 & meduc2<=15) if meduc2!=99 & meduc2!=.
	gen coll_plus = meduc2>=16 if meduc2!=99 & meduc2!=.

	* only count each mother once
	keep if totaleverborn==2 //keep at this parity

		
	* Table preliminaries
	preserve
		keep if totaleverborn==2
	cap erase "$work/Tables/xtable_1.tex"
	file open xtable_match using "$work/Tables/xtable_1.tex", write replace
	
		file write xtable_match ///
			"\begin{tabular}{l cccc}" _n ///
			"\toprule" _n ///
			"& \multicolumn{1}{c}{Matched} & \multicolumn{1}{c}{Unmatched} && \multicolumn{1}{c}{Difference} \\" _n ///
			"& \multicolumn{1}{c}{(1)} & \multicolumn{1}{c}{(2)} && \multicolumn{1}{c}{(3)}  \\" _n ///
			"\cline{2-5}" _n ///
		
			* label variables of interest
			label var mage2 "Age"
			label var mnativity "Native"
			label var ca_born "CA Born"
			label var less_than_hs "Less than HS"
			label var hs "HS Diploma"
			label var some_coll "Some College"
			label var coll_plus "BA or More"
		
			foreach var in mage2 mnativity ca_born less_than_hs hs some_coll coll_plus{
				
				* matched
				sum `var' if unlinked==0
				local m1 = "\multicolumn{1}{c}{" + string(`r(mean)', "%10.3fc") + "}"
				
				* unmatched
				sum `var' if unlinked==1
				local m2 = "\multicolumn{1}{c}{" + string(`r(mean)', "%10.3fc") + "}"
				
				* difference
				reg `var' unlinked
				store_regular3 unlinked
				local diff = "`r(beta)'`r(star)'"
				
				local label: variable label `var'
				file write xtable_match "`label' & `m1' & `m2' && `diff' \\" _n
			}	
		
		file write xtable_match "\hline" _n
		
		cou if unlinked==0
		local n1 = "\multicolumn{1}{c}{" + string(`r(N)', "%10.0fc") + "}"
		
		cou if unlinked==1
		local n2 = "\multicolumn{1}{c}{" + string(`r(N)', "%10.0fc") + "}"
		
		file write xtable_match "Observations & `n1' & `n2' && `' \\" 
		
	file write xtable_match _n ///
		"\bottomrule" _n ///
		"\end{tabular}" _n
	
	file close xtable_match
	macro drop xtable_match
	restore
		
*** now additionally restrict to just CA born mothers
	preserve
	keep if mbirthplace=="CA"
	keep if totaleverborn==2 //keep at this parity
	
	* Table preliminaries
	cap erase "$work/Tables/xtable_1_panel_b.tex"
	file open xtable_match using "$work/Tables/xtable_1_panel_b.tex", write replace
	
		file write xtable_match ///
			"\begin{tabular}{l cccc}" _n ///
			"\toprule" _n ///
			"& \multicolumn{1}{c}{Matched} & \multicolumn{1}{c}{Unmatched} && \multicolumn{1}{c}{Difference} \\" _n ///
			"& \multicolumn{1}{c}{(1)} & \multicolumn{1}{c}{(2)} && \multicolumn{1}{c}{(3)}  \\" _n ///
			"\cline{2-5}" _n ///
		
			* label variables of interest
			label var mage2 "Age"
			label var mnativity "Native"
			label var ca_born "CA Born"
			label var less_than_hs "Less than HS"
			label var hs "HS Diploma"
			label var some_coll "Some College"
			label var coll_plus "BA or More"
		
			foreach var in mage2 less_than_hs hs some_coll coll_plus{
				
				* matched
				sum `var' if unlinked==0
				local m1 = "\multicolumn{1}{c}{" + string(`r(mean)', "%10.3fc") + "}"
				
				* unmatched
				sum `var' if unlinked==1
				local m2 = "\multicolumn{1}{c}{" + string(`r(mean)', "%10.3fc") + "}"
				
				* difference
				reg `var' unlinked
				store_regular3 unlinked
				local diff = "`r(beta)'`r(star)'"
				
				local label: variable label `var'
				file write xtable_match "`label' & `m1' & `m2' && `diff' \\" _n
			}	
		
		file write xtable_match "\hline" _n
		
		cou if unlinked==0
		local n1 = "\multicolumn{1}{c}{" + string(`r(N)', "%10.0fc") + "}"
		
		cou if unlinked==1
		local n2 = "\multicolumn{1}{c}{" + string(`r(N)', "%10.0fc") + "}"
		
		file write xtable_match "Observations & `n1' & `n2' && `' \\" 
		
	file write xtable_match _n ///
		"\bottomrule" _n ///
		"\end{tabular}" _n
	
	file close xtable_match
	macro drop xtable_match	
	restore
	
*** now additionally restrict to just CA born mothers who also had a child born post-1998
	preserve
	keep if mbirthplace=="CA"
	keep if totaleverborn==2 //keep at this parity
	keep if birthyear>=1998
	
	* Table preliminaries
	cap erase "$work/Tables/xtable_1_panel_c.tex"
	file open xtable_match using "$work/Tables/xtable_1_panel_c.tex", write replace
	
		file write xtable_match ///
			"\begin{tabular}{l cccc}" _n ///
			"\toprule" _n ///
			"& \multicolumn{1}{c}{Matched} & \multicolumn{1}{c}{Unmatched} && \multicolumn{1}{c}{Difference} \\" _n ///
			"& \multicolumn{1}{c}{(1)} & \multicolumn{1}{c}{(2)} && \multicolumn{1}{c}{(3)}  \\" _n ///
			"\cline{2-5}" _n ///
		
			* label variables of interest
			label var mage2 "Age"
			label var mnativity "Native"
			label var ca_born "CA Born"
			label var less_than_hs "Less than HS"
			label var hs "HS Diploma"
			label var some_coll "Some College"
			label var coll_plus "BA or More"
		
			foreach var in mage2 less_than_hs hs some_coll coll_plus{
				
				* matched
				sum `var' if unlinked==0
				local m1 = "\multicolumn{1}{c}{" + string(`r(mean)', "%10.3fc") + "}"
				
				* unmatched
				sum `var' if unlinked==1
				local m2 = "\multicolumn{1}{c}{" + string(`r(mean)', "%10.3fc") + "}"
				
				* difference
				reg `var' unlinked
				store_regular3 unlinked
				local diff = "`r(beta)'`r(star)'"
				
				local label: variable label `var'
				file write xtable_match "`label' & `m1' & `m2' && `diff' \\" _n
			}	
		
		file write xtable_match "\hline" _n
		
		cou if unlinked==0
		local n1 = "\multicolumn{1}{c}{" + string(`r(N)', "%10.0fc") + "}"
		
		cou if unlinked==1
		local n2 = "\multicolumn{1}{c}{" + string(`r(N)', "%10.0fc") + "}"
		
		file write xtable_match "Observations & `n1' & `n2' && `' \\" 
		
	file write xtable_match _n ///
		"\bottomrule" _n ///
		"\end{tabular}" _n
	
	file close xtable_match
	macro drop xtable_match		
	restore
	
	
	
	
	
	
*****************************************************************************************************************
*** Appendix Figure 2, Panel B: Accounting for Naming Differences - Include Country x Birthorder Interactions ***	
*****************************************************************************************************************


	use "$work/compiled_indexalt.dta", replace
	setup_main
	
	
	* mbirthplace indicators
	gen mbpl = .
	replace mbpl = 1 if mbirthplace=="CH"
	replace mbpl = 6 if mbirthplace=="MX"
	replace mbpl = 8 if mbirthplace=="PI"
	replace mbpl = 10 if mbirthplace=="RE"
	replace mbpl = 12 if mbirthplace=="VN"
	replace mbpl = 13 if mnativity==1 // natives
	* group small sending countries into RE
	replace mbpl = 10 if mi(mbpl)
	
	* Due to computational constraints, we top code birthorder at 3 for this graph
	replace totaleverborn = 3 if totaleverborn>3
	
	* free up memory to estimate interactions
	drop mbirthplace firstname sex mrace_methnicity FBindex_fathers FBindex_mothers max_meduc ffirstname mnativity band2 band3
	compress

	* to combat memory issues
	keep if foreignborn==1
	* regress foreign born on mother's age indicators + band indicators + birth order parity indicators * country
	* omit robust standard errors due to computational constraints
	areg FBindex ind_mage_* band4-band7 i.totaleverborn##i.mbpl, abs(mgroup) coeflegend
		estimates store figurea2_fb
		
		
	use "$work/compiled_indexalt.dta", replace
	setup_main
	
	// note: no interactions because all are from US
	keep if foreignborn==0
	areg FBindex ind_mage* band4-band7 i.totaleverborn, abs(mgroup) vce(robust) coeflegend
		estimates store figurea2_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figurea2_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = _b[_cons] in 1
		replace se_native = _se[_cons] in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figurea2_fb
		replace coef_foreign = _b[_cons] in 1
		replace se_foreign = _se[_cons] in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(solid) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(on order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
		graph export "$work/Figures/figA2_b.eps", replace	
	
	
	
	
*************************************************
*** Appendix Figure 4: Reporting Coefficients ***	
*************************************************


use "$work/compiled_indexalt.dta", replace
	setup_main

	* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex ind_mage_* band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea4_fb
		
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0, abs(mgroup) vce(robust) coeflegend
		estimates store figurea4_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figurea4_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = 0 in 1
		replace se_native = 0 in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i']) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figurea4_fb
		replace coef_foreign = 0 in 1
		replace se_foreign = 0 in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i']) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(solid) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(-15(5)5) ///
			legend(on order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
		graph export "$work/Figures/figA4_b.eps", replace	

		
		
		
		
		
		
	
*****************************************************************************
********** Appendix Figure 5: Restrict to All Births in California **********
*****************************************************************************

	use "$work/compiled_all_in_ca_robust_indexalt.dta", replace
	setup_main
	
		* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex ind_mage_* band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea5_fb
		
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0, abs(mgroup) vce(robust) coeflegend
		estimates store figurea5_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figurea5_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = _b[_cons] in 1
		replace se_native = _se[_cons] in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figurea5_fb
		replace coef_foreign = _b[_cons] in 1
		replace se_foreign = _se[_cons] in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(solid) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
			
		graph export "$work/Figures/figA5.eps", replace			
		
		


*************************************************************
********** Appendix Figure 6: Using 1998-2015 Only **********	
*************************************************************

	use "$work/compiled_indexalt.dta", replace
	setup_main
	
	* restrict to years of interest
	keep if birthyear>=1998 & birthyear<=2015

		* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex ind_mage_* band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea6_fb
		
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0, abs(mgroup) vce(robust) coeflegend
		estimates store figurea6_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figurea6_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = _b[_cons] in 1
		replace se_native = _se[_cons] in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figurea6_fb
		replace coef_foreign = _b[_cons] in 1
		replace se_foreign = _se[_cons] in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(solid) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
		graph export "$work/Figures/figA6.eps", replace	
			


			

******************************************************************
********** Appendix Figure 7, Panel B: 3-year Age Bands **********
******************************************************************

	use "$work/compiled_indexalt.dta", replace
	setup_3yr
	
	
	* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex ind_mage_* band5-band12 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea7_fb
		
	areg FBindex ind_mage* band5-band12 i.totaleverborn if foreignborn==0, abs(mgroup) vce(robust) coeflegend
		estimates store figurea7_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figurea7_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = _b[_cons] in 1
		replace se_native = _se[_cons] in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figurea7_fb
		replace coef_foreign = _b[_cons] in 1
		replace se_foreign = _se[_cons] in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(solid) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
		graph export "$work/Figures/figA7_b.eps", replace	
			
		
					
			
			
			
			
************************************************************
********** Appendix Figure 9: Re-Weighted Results **********			
************************************************************			

	use "$work/compiled_indexalt.dta", replace
	setup_main
	
	* merge in sample weights
	merge m:1 mbirthplace using "$main/3work/ipums_2000_census_weights.dta", assert(1 3) nogen
	
	* create new sample weight
	* for each sending country, scale the baseline weight by the fraction (census/sample)
	* weights are 1 for nativeborn
	gen weight = 1	
	foreach sc in "CH" "CN" "CU" "GU" "JA" "MX" "PI" "PR" "RE" "VI" "VN"{
		di "`sc'"
		sum ipums_weight if mbirthplace=="`sc'"
		local iw = `r(mean)'
		qui gen temp = mbirthplace=="`sc'"
		sum temp
		local sw = `r(mean)'
		replace weight = weight*(`iw'/`sw') if mbirthplace=="`sc'"
		cap drop temp
	}


	* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex ind_mage_* band4-band7 i.totaleverborn if foreignborn==1 [aw=weight], abs(mgroup) vce(robust) coeflegend
		estimates store figurea9_fb
		
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0 [aw=weight], abs(mgroup) vce(robust) coeflegend
		estimates store figurea9_n
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = 20 in 1
		gen coef_native = .
			gen se_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_foreign = .
			gen se_foreign = .
			gen lb_foreign = .
			gen ub_foreign = .
		
		estimates restore figurea9_n
		local i = 1
		
		forvalues age = 21(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		* native-born
		replace coef_native = _b[_cons] in 1
		replace se_native = _se[_cons] in 1
		replace lb_native = coef_native - 1.96*se_native in 1
		replace ub_native = coef_native + 1.96*se_native in 1
		
		forvalues i = 21(1)40{
			replace coef_native = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_native = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_native = coef_native - 1.96*se_native if MomsAge==`i'
			replace ub_native = coef_native + 1.96*se_native if MomsAge==`i'
		}
		
		* foreign-born
		estimates restore figurea9_fb
		replace coef_foreign = _b[_cons] in 1
		replace se_foreign = _se[_cons] in 1
		replace lb_foreign = coef_foreign - 1.96*se_foreign in 1
		replace ub_foreign = coef_foreign + 1.96*se_foreign in 1
		
		forvalues i = 21(1)40{
			replace coef_foreign = (_b[ind_mage_`i'] + _b[_cons]) if MomsAge==`i'
			replace se_foreign = _se[ind_mage_`i'] if MomsAge==`i'
			replace lb_foreign = coef_foreign - 1.96*se_native if MomsAge==`i'
			replace ub_foreign = coef_foreign + 1.96*se_native if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_foreign MomsAge, mc(black) lc(black) m(o)) ///
			(rcap lb_foreign ub_foreign MomsAge, lc(black)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(solid) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(order(1 3) label(1 "Foreign-born mother") label(3 "US-born mother") position(6) rows(1))
			
		graph export "$work/Figures/figA9_b.eps", replace	







****************************************************************************
********** Appendix Figure 11, Panel B: Results by Sons/Daughters **********
****************************************************************************

	use "$work/compiled_indexalt.dta", replace
	setup_main
		
	* Native coefficients	
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0, abs(mgroup) vce(robust) coeflegend
		estimates store figurea11_base

	* Sons
	areg FBindex ind_mage* band4-band7 i.totaleverborn if sex==1 & foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea11_sons
		
	* Daughters
	areg FBindex ind_mage* band4-band7 i.totaleverborn if sex==2 & foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea11_daughters	

		
		
		* write betas to locals for graphing
		estimates restore figurea11_base
		local native_20 = _b[_cons]
		local native_se_20 = _se[_cons]
		forvalues i = 21(1)40{
			local native_`i' = (_b[ind_mage_`i'] + _b[_cons])
			local native_se_`i' = (_se[ind_mage_`i'])
		}
		
		estimates restore figurea11_sons
		local sons_20 = _b[_cons]
		local sons_se_20 = _se[_cons]
		forvalues i = 21(1)40{
			local sons_`i' = (_b[ind_mage_`i'] + _b[_cons])
			local sons_se_`i' = (_se[ind_mage_`i'])
		}
		
		estimates restore figurea11_daughters
		local daughters_20 = _b[_cons]
		local daughters_se_20 = _se[_cons]
		forvalues i = 21(1)40{
			local daughters_`i' = (_b[ind_mage_`i'] + _b[_cons])
			local daughters_se_`i' = (_se[ind_mage_`i'])
		}
		
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = .
		gen coef_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_sons = .
			gen lb_sons = .
			gen ub_sons = .
		gen coef_daughters = .
			gen lb_daughters = .
			gen ub_daughters = .
		local i = 0
	
		forvalues age = 20(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		forvalues i = 20(1)40{
			replace coef_native = `native_`i'' if MomsAge==`i'
				replace lb_native = `native_`i'' - 1.96*`native_se_`i'' if MomsAge==`i'
				replace ub_native = `native_`i'' + 1.96*`native_se_`i'' if MomsAge==`i'
			
			replace coef_sons = `sons_`i'' if MomsAge==`i'
				replace lb_sons = `sons_`i'' - 1.96*`sons_se_`i'' if MomsAge==`i'
				replace ub_sons = `sons_`i'' + 1.96*`sons_se_`i'' if MomsAge==`i'
			
			replace coef_daughters = `daughters_`i'' if MomsAge==`i'
				replace lb_daughters = `daughters_`i'' - 1.96*`daughters_se_`i'' if MomsAge==`i'
				replace ub_daughters = `daughters_`i'' + 1.96*`daughters_se_`i'' if MomsAge==`i'
		}
		
		* graph
		twoway (connected coef_sons MomsAge, mc(black) lc(black) lp(solid) m(o)) ///
			(rcap lb_sons ub_sons MomsAge, lc(black)) ///
			(connected coef_daughters MomsAge, mc(gs12) lc(gs12) lp(solid) m(o)) ///
			(rcap lb_daughters ub_daughters MomsAge, lc(gs12)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(dash) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)60) ///
			legend(order(1 3 5) label(1 "Foreign-born mother, sons") label(3 "Foreign-born mother, daughters") label(5 "US-born mother, all children") position(6) rows(1))
			
			
		graph export "$work/Figures/figA11_B.eps", replace	
		



		

***********************************************************************************************
********** Appendix Figure 12, Panel B: Results by Sons/Daughters - Second Gen Index **********
***********************************************************************************************

	use "$work/compiled.dta", replace
	setup_main
		
	* Note: We omit robust standard errors due to memory constraints	
	* Native coefficients	
	areg FBindex ind_mage* band4-band7 i.totaleverborn if foreignborn==0, abs(mgroup) coeflegend
		estimates store figurea12_base

	* Sons
	areg FBindex ind_mage* band4-band7 i.totaleverborn if sex==1 & foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea12_sons
		
	* Daughters
	areg FBindex ind_mage* band4-band7 i.totaleverborn if sex==2 & foreignborn==1, abs(mgroup) vce(robust) coeflegend
		estimates store figurea12_daughters	

		
		
		* write betas to locals for graphing
		estimates restore figurea12_base
		local native_20 = _b[_cons]
		local native_se_20 = _se[_cons]
		forvalues i = 21(1)40{
			local native_`i' = (_b[ind_mage_`i'] + _b[_cons])
			local native_se_`i' = (_se[ind_mage_`i'])
		}
		
		estimates restore figurea12_sons
		local sons_20 = _b[_cons]
		local sons_se_20 = _se[_cons]
		forvalues i = 21(1)40{
			local sons_`i' = (_b[ind_mage_`i'] + _b[_cons])
			local sons_se_`i' = (_se[ind_mage_`i'])
		}
		
		estimates restore figurea12_daughters
		local daughters_20 = _b[_cons]
		local daughters_se_20 = _se[_cons]
		forvalues i = 21(1)40{
			local daughters_`i' = (_b[ind_mage_`i'] + _b[_cons])
			local daughters_se_`i' = (_se[ind_mage_`i'])
		}
		
		
		* write output to figure
		clear
		set obs 21
		gen MomsAge = .
		gen coef_native = .
			gen lb_native = .
			gen ub_native = .
		gen coef_sons = .
			gen lb_sons = .
			gen ub_sons = .
		gen coef_daughters = .
			gen lb_daughters = .
			gen ub_daughters = .
		local i = 0
	
		forvalues age = 20(1)40{
			local ++ i
			replace MomsAge = `age' in `i'
		}
		
		forvalues i = 20(1)40{
			replace coef_native = `native_`i'' if MomsAge==`i'
				replace lb_native = `native_`i'' - 1.96*`native_se_`i'' if MomsAge==`i'
				replace ub_native = `native_`i'' + 1.96*`native_se_`i'' if MomsAge==`i'
			
			replace coef_sons = `sons_`i'' if MomsAge==`i'
				replace lb_sons = `sons_`i'' - 1.96*`sons_se_`i'' if MomsAge==`i'
				replace ub_sons = `sons_`i'' + 1.96*`sons_se_`i'' if MomsAge==`i'
			
			replace coef_daughters = `daughters_`i'' if MomsAge==`i'
				replace lb_daughters = `daughters_`i'' - 1.96*`daughters_se_`i'' if MomsAge==`i'
				replace ub_daughters = `daughters_`i'' + 1.96*`daughters_se_`i'' if MomsAge==`i'
		}
		
		
		
		* graph (alt vers)
		twoway (connected coef_sons MomsAge, mc(black) lc(black) lp(solid) m(o)) ///
			(rcap lb_sons ub_sons MomsAge, lc(black)) ///
			(connected coef_daughters MomsAge, mc(gs12) lc(gs12) lp(solid) m(o)) ///
			(rcap lb_daughters ub_daughters MomsAge, lc(gs12)) ///
			(connected coef_native MomsAge, mc(gray) lc(gray) lp(dash) m(o)) ///
			(rcap lb_native ub_native MomsAge, lc(gray)), ///
			xtitle("Mother's Age at Birth of Child") ///
			ytitle("Foreignness Index") ///
			xlab(20(1)40, nogrid) ///
			ylab(20(10)70) ///
			legend(order(1 3 5) label(1 "Foreign-born mother, sons") label(3 "Foreign-born mother, daughters") label(5 "US-born mother, all children") position(6) rows(1))
			
			
		graph export "$work/Figures/figA12_B.eps", replace	
		
		
		
		
**************************************************************
********** Appendix Table 5: Robustness on F-index **********		
**************************************************************

	cap erase "$work/Tables/xtable_findex_robust.tex"
	file open xtable_robust using "$work/Tables/xtable_findex_robust.tex", write replace
	
		file write xtable_robust ///
			"\begin{tabular}{l c}" _n ///
			"\toprule" _n ///
			" \multicolumn{2}{c}{Dependent Variable: Foreignness Index}  \\" _n ///
			"\cline{1-2}" _n ///
			"& \multicolumn{1}{c}{(1)} \\" _n ///
			"\cline{2-2}" _n ///	
		
		
		* Baseline linear result
		use "$work/compiled_indexalt.dta", replace
		setup_main

		areg FBindex mage band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
			store_regular3 mage
			local b = "`r(beta)'`r(star)'"
			local se = "`r(se)'" 
			
			local n = string(`e(N)', "%10.0fc")
			
		file write xtable_robust "Baseline & `b' \\" _n ///
			"\emph{N} = `n' & `se' \\" _n
			
		file write xtable_robust "\addlinespace" _n
		
		
		* Country-specific result
		use "$work/compiled_country_specific_indexalt.dta", replace
		setup_main

		areg FBindex mage band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
			store_regular3 mage
			local b = "`r(beta)'`r(star)'"
			local se = "`r(se)'" 
			
			local n = string(`e(N)', "%10.0fc")
			
		file write xtable_robust "Country-Specific F-index & `b' \\" _n ///
			"\emph{N} = `n' & `se' \\" _n
			
		file write xtable_robust "\addlinespace" _n
		
	
		* NYSIIS Adjustments
		use "$work/compiled_nysiis_indexalt.dta", replace
		setup_main

		areg FBindex mage band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) coeflegend
			store_regular3 mage
			local b = "`r(beta)'`r(star)'"
			local se = "`r(se)'" 
			
			local n =  string(`e(N)', "%10.0fc")
			
		file write xtable_robust "Adjust names with NYSIIS & `b' \\" _n ///
			"\emph{N} = `n' & `se' \\" _n
			
		file write xtable_robust "\addlinespace" _n	
	

		file write xtable_robust ///
			"\bottomrule" _n ///
			"\end{tabular}" _n
		
		file close xtable_robust
		macro drop xtable_robust
				
		
		
		
		
*******************************************
*** One Linear Estimate of Assimilation ***		
*******************************************		
		
		
use "$work/compiled_indexalt.dta", replace
	setup_main

	* regress foreign born on mother's age indicators + band indicators + birth order parity indicators
	areg FBindex mage band4-band7 i.totaleverborn if foreignborn==1, abs(mgroup) vce(robust) 
	
			
		
		
******************************************
********** Baseline Difference ***********		
******************************************

	use "$work/compiled_indexalt.dta", replace
	setup_main
		
		* create interactions between foreign born and age
		forvalues i = 21(1)40{
			gen ind_fb_mage_`i' = ind_mage_`i' * foreignborn
		}	
		compress

		reg FBindex ind_* foreignborn band4-band7 i.totaleverborn, vce(robust) coeflegend	
		
		* foreignborn is difference at age 20
		di _b[foreignborn]		
			
		
		
		
		
		
