% This file computes chi-squared statistics and p-values for the test of
% a constant recession probability across each of the chosen horizons. It 
% requires the following files (see ReadMe.pdf for descriptions):
%
%   - rcindCreate.m
%   - fnRegr.m
%   - NewWest.m
%   - fnRegrChisq.m
%   - simrcind.m
%   - recdats.mat
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Initialize

clearvars

%% Options

fdat = 1946;            % first date to use (min = 1857.25)
ldat = 2017.25;         % last date to use

x = 5;                  % vector of window sizes; in paper, x=5
kvc = [38,56];          % vector of horizons to test for constant prob.; 
                            % in paper, [38,56] or [36:39,55:58]

bt = 0;         % 1 = bootstrap covariance matrix; 0 = asymptotic Newey-West
N = 10000;      % number of bootstrap replications

%%

load recdats        % load data file

% This block finds the first date to consider (start date of first
% recession following fdat0)
pkdt = pkdt((pkdt>=fdat)&(pkdt<=ldat));    % drop peak dates outside of specified date range
fpkdt = pkdt(1);                           % date of first peak in sample period
trdt = trdt((trdt>=fpkdt)&(trdt<=ldat));   % drop trough dates outside of specified date range

reclen = 4*(trdt-pkdt);       % vector of recession lengths in quarters
explen = 4*[pkdt(2:end)-trdt(1:end-1);ldat-trdt(end)+.25];  % vector of expansion lengths in quarters

dtvc = (fdat:.25:ldat)';       % vector of quarters spanning sample period
T = numel(dtvc);                % number of quarters in sample period
nx = numel(x);                  % number of window sizes
nk = numel(kvc);                % number of horizons

% call function to convert sequence of expansion/recession lengths to 0-1
% recession indicators
rcind = rcindCreate(reclen,explen,T,4*(fpkdt-fdat)+2);

%% Compute covariance matrix

% allocate memory
rpr = zeros(nk,nx);     % rpr(q,j) is point estimate of recession probability
                            % at horizon kvc(q) for window x(j)
rprpmn = zeros(nx,1);   % rprmn(j) is average point estimate for window x(j)
                            % across all horizons in kvc (null hypothesis
                            % value for constant null)
Sig = zeros(nk,nk,nx);  % Sig(:,:,j) is covariance matrix of point estimates
                            % across the horizons in kvc for window x(j)

% get probabilities and covariance matrices
if bt == 0          % if using asymptotic estimator
    for j = 1:nx        % for each window size
        
        % call function to get point estimates and covariance matrix
        [rprj,Sigj] = fnRegrChisq(rcind,kvc,x(j));
        
        % load results into arrays to keep
        rpr(:,j) = rprj;
        Sig(:,:,j) = Sigj;
        rprpmn(j) = mean(rprj);
    end
else                % otherwise if using bootstrap method
    
    % call function to bootstrap N recession/expansion histories of same
    % length as data history
    rcindsim = simrcind(T,N,reclen,explen); 
    
    % rprsim(q,i,j) is point estimate of recession probability at horizon
    % kvc(q) for simulated history i with window x(j)
    rprsim = zeros(nk,N,nx);    
	
    for j = 1:nx        % for each window size
        
        % call function to obtain recession probability point estimates for
        % actual data
        rpr(:,j) = fnRegr(rcind,[],kvc,x(j));
        rprpmn(j) = mean(rpr(:,j));
                
        for i = 1:N         % for each simulated history
            
            % call function to obtain recession probability point estimates for
            % simulated history i
            rprsim(:,i,j) = fnRegr(rcindsim(:,i),[],kvc,x(j));
        end
        
        Sig(:,:,j) = cov(rprsim(:,:,j)');     % covariance matrix derived from bootstraps
    end
end

%% Display results

% set up display cell
dispcl = cell(nx+1,5);
dispcl{1,1} = 'Window sz.';
dispcl{1,2} = 'statistic';
dispcl{1,3} = 'DOF';
dispcl{1,4} = 'p-val.';
dispcl{1,5} = '# k''s';

for j = 1:nx        % for each window size
    
    Sigj = Sig(:,:,j);      % cov matrix for j-th window
    
    cSigj = chol(Sigj,'lower');         % Cholesky square root
    eta = cSigj\(rpr(:,j)-rprpmn(j));   % vector of N(0,1) RV's under null
    ch2stat = sum(eta.^2);              % chi2 RV under null
    chi2p = 1-chi2cdf(ch2stat,nk);      % p-value
    
    % load results into display cell
    dispcl{j+1,1} = x(j);
    dispcl{j+1,2} = ch2stat;
    dispcl{j+1,3} = nk;
    dispcl{j+1,4} = chi2p;
    dispcl{j+1,5} = nk;
end

disp(dispcl)













