% This file produces point estimates and confidence intervals for the
% regression probability figures. It requires the following files (see
% ReadMe.pdf for descriptions):
%
%   - rcindCreate.m
%   - fnRegr.m
%   - NewWest.m
%   - recdats.mat
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Initialize

clearvars

%% Options

fdat = 1946;            % first date to use (min = 1857.25)
ldat = 2017.25;         % last date to use

x = (3:5);              % row vector of window sizes (1 figure for each)
kmn = 12;               % shortest horizon (k) to plot
kmx = 90;               % longest horizon (k) to plot

CIpct = [90;80;66];     % column vector of CI sizes (percentage), in decreasing order

ffign = 1;                  % first figure number to use
fsz = 24;                   % font size for axis/legend
lnwd = 6;                   % line width for plots
figpos = .8*[0 0 1058 817]; % figure position

%% Load in peak/trough dates and create 0-1 recession indicators

load recdats        % load data file

% This block finds the first date to consider (start date of first
% recession following fdat0)
pkdt = pkdt((pkdt>=fdat)&(pkdt<=ldat));    % drop peak dates outside of specified date range
fpkdt = pkdt(1);                           % date of first peak in sample period
trdt = trdt((trdt>=fpkdt)&(trdt<=ldat));   % drop trough dates outside of specified date range

reclen = 4*(trdt-pkdt);       % vector of recession lengths in quarters
explen = 4*[pkdt(2:end)-trdt(1:end-1);ldat-trdt(end)+.25];  % vector of expansion lengths in quarters

dtvc = (fdat:.25:ldat)';       % vector of quarters spanning sample period
T = numel(dtvc);                % number of quarters in sample period
nx = numel(x);                  % number of window sizes
kvc = (kmn:kmx);                % vector of horizons to plot
nk = numel(kvc);                % number of horizons to plot

% call function to convert sequence of expansion/recession lengths to 0-1
% recession indicators
rcind = rcindCreate(reclen,explen,T,4*(fpkdt-fdat)+2);

%% Run regressions and compute CIs

nCI = numel(CIpct);                     % number of CIs
prctl = [(100-CIpct)/2,(100+CIpct)/2];  % get lower/upper bound percentiles

% rpr(k,j) will hold recession probability point estimates at horizon
% kvc(k), window size x(j)
rpr = zeros(nk,nx);                     % allocate memory

% bnds(k,:,m,j) will hold lower/upper bounds of CI for horizon kvc(k),
% window size x(j), interval size CIpct(m)
bnds = zeros(nk,2,nCI,nx);              % allocate memory

for j = 1:nx        % for each window size
    
    % call function to get point estimates and CIs
    [rprj,bndsj] = fnRegr(rcind,prctl,kvc,x(j));
    
    % load results into variabes
    rpr(:,j) = rprj;
    bnds(:,:,:,j) = bndsj;
end
    
%% Plots

fign = ffign;       % initialize figure number to use

CIclr = [0.8;.65;.5]*ones(1,3);     % colors to use for bands

for j = 1:nx    % for each window size
    
    % initialize figure
    figure(fign)
    fign = fign+1;
    set(gcf, 'Position', figpos)        % set location and size
    clf
    hold all
    
    % variables to track min/max of plotted values for setting y-axis limits
    mxdt = max(rpr(:,j));
    mndt = min(rpr(:,j));
    
    ah = zeros(1,nCI);  % vector to hold area plot handles
    for m = 1:nCI       % for each CI size
        sbndsj = bnds(:,:,m,j);             % extract CI bands
        
        % updaate mins/maxes
        mxdt = max(mxdt,max(sbndsj(:)));    
        mndt = min(mndt,min(sbndsj(:)));
        
        % plot CI bands
        ah(m) = fill([kmn:kmx,fliplr(kmn:kmx)],[sbndsj(:,1)',fliplr(sbndsj(:,2)')],CIclr(m,:));
    end
    
    set(ah,'EdgeColor','none')      % remove edges from bands
    
    % plot point estimates
    hdat = plot(kvc,rpr(:,j),'-','LineWidth',lnwd,'Color','k');
    
    % set axis limits
    dtdff = .01*(mxdt-mndt);
    yl = [max(mndt-dtdff,0),min(mxdt+dtdff,1)];
    ylim(yl)
    xlim([kmn,kmx])
    
    % labels and formatting
    ylabel('Probability of Recession','FontName','Times','FontSize',fsz,'Interpreter','latex')
    xlabel('$k$ (Quarters Since Recession)','FontName','Times','FontSize',fsz,'Interpreter','latex')
    title(['Window $x=' num2str(x(j)) '$'],'FontName','Times','FontSize',fsz,'Interpreter','latex')
	set(gca,'FontSize',.8*fsz)

    % legend
    lgcl = cell(1,nCI);
    for m = 1:nCI
        lgcl{m} = [num2str(CIpct(nCI+1-m)) '\% interval'];
    end    
    hleg = legend(fliplr(ah),lgcl,'Location','north','FontName','Times','FontSize',fsz-3,'Interpreter','latex');
    
end




