#include <math.h>
#include <stdlib.h>
#define dgemm dgemm_
#define dgesv dgesv_
#include "mex.h"
#include "blas.h"
#include "lapack.h"
#include <algorithm>
#include <random>


void mexFunction(int nlhs, mxArray *plhs[],
                int nrhs, const mxArray *prhs[])
{
    // output pointer definitions
    #define Stsim_OUT       plhs[0]
    #define tcut_OUT        plhs[1]
    
    // input pointer definitions
    #define T_IN            prhs[0]
    #define St0_IN          prhs[1]
    #define shk_IN          prhs[2]
    #define coefs_IN        prhs[3]
    #define expns_IN        prhs[4]
    #define MP_IN           prhs[5]
    #define ctoff_IN        prhs[6]
    #define ntrm_IN         prhs[7]
    
    // other macros
    #define nSt             3
    #define iX              0
    #define il              1
    #define inu             2
            
    // macro to read in scalar parameters
    #define prscl(PAR) (mxGetScalar(mxGetField(MP_IN,0,PAR)))
    // macro to read in array parameters
    #define prarr(PAR) (mxGetPr(mxGetField(MP_IN,0,PAR)))

    // scalar parameters
    int T = (int)mxGetScalar(T_IN);         // # of periods for simulation
	double ctoff = mxGetScalar(ctoff_IN);   // cutoff for explosion in l
    int ntrm = (int)mxGetScalar(ntrm_IN);   // number of terms in polynomial
    double del = prscl("del");              // depreciation rate
    double psi = prscl("psi");              // proportion of purchases entering durables
    double rho = prscl("rho");              // persistence of shock
    double sig = prscl("sig");              // s.d. of shock innovation
    
    // array parameters
	double *St0 = mxGetPr(St0_IN);          // initial state vector
	double *shk = mxGetPr(shk_IN);	// vector of innovations
    double *coefs = mxGetPr(coefs_IN);      // vector of coefficients in evolution
    double *expns = mxGetPr(expns_IN);      // matrix of exponents on state variables
    
    // initialize outs    
    Stsim_OUT = mxCreateDoubleMatrix(nSt,T,mxREAL);
    tcut_OUT = mxCreateDoubleMatrix(1,1,mxREAL);
    double *Stsim = mxGetPr(Stsim_OUT);
    double *tcut = mxGetPr(tcut_OUT);
    *tcut = 0.0;
    

    // other scalars
    int ii = 0, kk = 0, t = 0;
    int szdb = sizeof(double);
	int szin = sizeof(int);
    double tmp = 0.0;
    
    // other arrays
    double lt = 0.0;
    int *exps = (int*)mxMalloc(ntrm*nSt * szin);
    
    // initialize exps
    for(ii=0; ii<ntrm; ii++)    {
        for(kk=0; kk<nSt; kk++)      {
            exps[ntrm*kk+ii] = (int)expns[ntrm*kk+ii];
        }
    }
    
    // initial values
    for(ii=0; ii<ntrm; ii++)    {
        tmp = coefs[ii];
        for(kk=0; kk<nSt; kk++)      {
            tmp = tmp*pow(St0[kk],exps[ntrm*kk+ii]);
        }
        lt = lt + tmp;
    }
	
    Stsim[iX] = (1-del)*St0[iX] + psi*lt;
    Stsim[il] = lt;
    Stsim[inu] = rho*St0[inu] + sig*shk[0];

    if(ctoff > 0)	{
        if(fabs (lt) > ctoff)   {
            *tcut = 1.0;
            return;
        }
    }
    
    
    // Remaining simulation
    for(t=0; t<T-1; t++)     {
        lt = 0.0;
        for(ii=0; ii<ntrm; ii++)    {
            tmp = coefs[ii];
            for(kk=0; kk<nSt; kk++)      {
                tmp = tmp*pow(Stsim[nSt*t+kk],exps[ntrm*kk+ii]);
            }
            lt = lt + tmp;
        }
        
        Stsim[nSt*(t+1)+iX] = (1-del)*Stsim[nSt*t+iX] + psi*lt;
        Stsim[nSt*(t+1)+il] = lt;
        Stsim[nSt*(t+1)+inu] = rho*Stsim[nSt*t+inu] + sig*shk[t+1];
        
        if(ctoff > 0)	{
            if(fabs (lt) > ctoff)   {
                *tcut = 1.0 + t + (ctoff-fabs(Stsim[nSt*t+il]))/(fabs(lt)-fabs(Stsim[nSt*t+il]));
                return;
            }
        }
    }
    
 
    
    mxFree(exps);
    return;
}