function NKRF_Derivs_Full(ord,expns,cmbs)
%
% This function generates the files that are used to iteratively obtain the
% higher-order derivatives of the perturbation solution.
% 
% Inputs:
%   ord         order of perturbation
%   expns       matrix of monomial exponents (see TaylExp.m)
%   cmbs        number of times each monomial appears in Taylor expansion
%                   (see TaylExp.m)
%
% The function does not return any outputs, but rather automatically writes
% the necessary files.
%
% NOTE: Requires MATLAB Symbolic Toolbox.
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Set up 

% variables
syms X Xp1 lm1 l lp1 nu ep1 sig
assume([X Xp1 lm1 l lp1 nu ep1 sig],'real')

% "state" vector
Y = [X;lm1;nu;sig];
nst = numel(Y);        % number of state variables

% parameters
syms del xi al2 al1 rho
psi = sym('psi');
prs = [del psi xi al2 al1 rho];
assume(prs,'real')

%% Generate symbolic representation of non-linear function 'th' as first
% 'ord' terms of a Taylor series. Also generate symbolic vector
% of the derivatives of th.

th = sym(0);
nDth = 3;
Dth = sym('Dth',[nDth,1]);
for j = 1:nDth
    % create symbolic variable for j-th order derivative
    eval(['Dth' num2str(j) ' = sym(''Dth' num2str(j) ''',''real'');'])
    
    % load symbolic variable into symbolic vector
    eval(['Dth(j) = Dth' num2str(j) ';'])
    
    % add j-th order term to Taylor expansion
    th = th + Dth(j)*l^j/factorial(j);
end

%% Construct 'ord'-th order Taylor expansion of policy function

ntrm = size(expns,1);   % number of monomials in n-th order Taylor expansion of policy function

dPhi = sym('dPhi',[ntrm,1],'real');    % vector of partial derivatives of Phi

Phi = sym(0);
for j = 1:ntrm      % for each monomial
    % construct variable name for corresponding derivative as dPhi####
    % where k-th # is order of derivative in k-th state variable
    cfnm = 'dPhi';
    for k = 1:nst
        cfnm = [cfnm num2str(expns(j,k))];
    end
    
    % create symbolic variable from variable name
    eval([cfnm ' = sym(''' cfnm ''',''real'');'])
    
    % load symbolic variable into symbolic vector
    eval(['dPhi(j) = ' cfnm ';'])
    
    % add corresponding monomial to Taylor expansion
    Phi = Phi + (dPhi(j)*cmbs(j)/factorial(sum(expns(j,:))))*prod(Y.^sym(expns(j,:)'));
    
end

%% Compute moments of epsilon(t+1)

% MGF for normal distribution
t = sym('t','real');
nmgf = symfun(exp(t^2/2),t);

% vector holding moments of epsilon(t+1)
epmom = sym('Eeps',[ord,1]);
for j = 1:ord
    % j-th derivative of MGF evaluated at zero
    epmom(j) = subs(diff(nmgf,t,j),t,0);
end

%% Construct Euler equation

Xp1 = (1-del)*X + psi*Phi;              % capital evolution
lp1 = subs(Phi,[X,lm1,nu],[Xp1,l,rho*nu+sig*ep1]);    % implied value of l(t+1)

eulfn = xi*l - xi*al2*lm1 + xi*al1*X - xi*th - xi*nu - lp1;
eulfn = subs(eulfn,l,Phi);      % substitute Phi in for l(t)
eul = (eulfn == 0); % Euler equation

%% Construct E_t[l(t+1)]

EPhi = subs(lp1,l,Phi);
% Deal with expectations of epsilon(t+1):
for p = ord:-1:1                  % for each exponent p on eps(t+1) starting with the highest
    EPhi = subs(EPhi,ep1^p,epmom(p));     % substitue in value of E[eps(t+1)^p]
end

%% Get derivatives evaluated at (0,0,0)

inds = cell(ord,1);   % j-th element holds indices of j-th-order terms in 'expns'
Deul = cell(ord,1);    % j-th element holds vector j-th-order derivatives of Euler equation

dff0 = eul;
for j = 1:ord
    inds{j} = find(sum(expns,2)==j);    % indices of j-th-order terms
    nordj = numel(inds{j});             % number of j-th-order terms
    Deulj = sym('Deulj',[nordj,1]);     % vector to hold j-th-order derivatives of Euler
    expnsj = expns(inds{j},:);          % exponents of j-th-order terms
    dff1 = sym('dff1',[nordj,1]);
    
    for k = 1:nordj                     % for each j-th-order term
        
        expnsjk = expnsj(k,:);          % vector of exponents for this term
        
        % Here we construct a variable 'expdff' that holds the number of
        % additional derivatives that need to be taken starting from the
        % (j-1)-th-order term that is closest to this term among those with
        % exponents not greater than the ones on this term. The idea is to
        % minimize the number of derivatives that need to be taken by
        % starting from a lower-order term that's close.
        if j == 1                       % if this is a first-order term
            
            expdff = expnsjk;               % Set number of derivatives to take, where
            dff1(k) = eul;                  % we start from the full Euler equation.
            
        else                            % otherwise
            
            expnsjm1 = expns(inds{j-1},:);  % get matrix of exponents on (j-1)-th-order terms
            
            % create matrix, each of whose rows is the current vector of exponents
            repexpnstjk = repmat(expnsjk,numel(inds{j-1}),1);
            % compute differences between current and (j-1)-th-order exponents
            dffexp = repexpnstjk - expnsjm1;
            % Find logical indices of (j-1)-th-order terms that have
            % exponents no greater than current term. Note that, since
            % every element of 'expnsjm1' corresponds to a (j-1)-th-order
            % term, every element corresponding to a value of 1 in kpind
            % is only one derivative short of the current term.
            kpind = prod(dffexp >= 0,2);
            % get linear index of first such element 
            kpind = find(kpind,1);
            
            expdff = expnsjk - expnsjm1(kpind,:);   % Vector of number of derivatives to take
            dff1(k) = dff0(kpind);                  % Starting from closest (j-1)-th order term
        end
        
        m = find(expdff>0);
        dff1(k) = diff(dff1(k),Y(m));   % differentiate (j-1)-th-order term w.r.t.
                                            % appropriate state variable
        
        dff = subs(dff1(k),Y,zeros(nst,1));    % evaluate derivative at 0
        
        % Deal with expectations of epsilon(t+1):
        for p = j:-1:1                  % for each exponent p on eps(t+1) starting with the highest
            dff = subs(dff,ep1^p,epmom(p));     % substitue in value of E[eps(t+1)^p]
        end
        
        % load final expression into vector of j-th-order derivatives
        Deulj(k) = dff;
    end
    
    % load vector of j-th-order derivatives into j-th element of cell array
    Deul{j} = Deulj;
    
    dff0 = dff1;
end


%% Construct functions to obtain j-th-order derivatives from (j-1)-th- and
%  lower-order derivatives.

dPhijm1 = dPhi(inds{1});    % vector of lower-order derivatives to pass in

for j = 2:ord               % for each order >= 2
    flnm = ['DervFll_ord' num2str(j) '_fn'];   % name of function
    dPhij = dPhi(inds{j});                  % vector of j-th-order derivatives of policy function
    
    % construct matrices A and B in expression A*dPhij = B that can then be
    % used to solve for dPhij
    
    [A,B] = equationsToMatrix(Deul{j}, dPhij);
    
    % create m-function that takes parameters and lower-order derivatives
    % as inputs and returns A and B
    matlabFunction(A,B,'File',flnm,'Vars',{dPhijm1,Dth,prs},'Outputs',{'A','B'});
	
	dPhijm1 = [dPhijm1;dPhij];  % update vector of lower-order derivatives to pass in
end











